/*
 * Copyright (C) 2000 - 2024 Silverpeas
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * As a special exception to the terms and conditions of version 3.0 of
 * the GPL, you may redistribute this Program in connection with Free/Libre
 * Open Source Software ("FLOSS") applications as described in Silverpeas's
 * FLOSS exception.  You should have received a copy of the text describing
 * the FLOSS exception, and it is also available here:
 * "https://www.silverpeas.org/legal/floss_exception.html"
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.silverpeas.core.web.util.viewgenerator.html.datereminder;

import java.util.Date;
import org.silverpeas.core.util.DateUtil;
import org.silverpeas.kernel.bundle.LocalizationBundle;
import org.silverpeas.kernel.bundle.ResourceLocator;
import org.silverpeas.core.contribution.publication.model.PublicationDetail;
import org.silverpeas.core.persistence.EntityReference;
import org.silverpeas.core.datereminder.persistence.service.DateReminderServiceProvider;
import org.silverpeas.core.datereminder.persistence.service.PersistentDateReminderService;
import org.silverpeas.core.datereminder.persistence.PersistentResourceDateReminder;
import org.silverpeas.core.datereminder.persistence.DateReminderDetail;
import org.silverpeas.core.contribution.publication.datereminder.PublicationNoteReference;

/**
 * The widget date reminder
 *
 * @author Cécile Bonin
 */
public class DateReminder {

  private String resourceId;
  private String resourceType;
  private String userId;
  private String language;
  private LocalizationBundle messages;

  public DateReminder(String resourceId,
      String resourceType, String userId, String language) {
    this.resourceId = resourceId;
    this.resourceType = resourceType;
    this.userId = userId;
    this.language = language;
    this.messages = ResourceLocator
        .getLocalizationBundle("org.silverpeas.dateReminder.multilang.dateReminder", language);
  }

  /**
   * Sets the unique identifier of the resource that use DateReminder.
   *
   * @param resourceId the unique identifier of the resource.
   */
  public void setResourceId(String resourceId) {
    this.resourceId = resourceId;
  }

  /**
   * Sets the type of the resource that use DateReminder.
   *
   * @param resourceType the type of the resource.
   */
  public void setResourceType(String resourceType) {
    this.resourceType = resourceType;
  }

  /**
   * Sets the unique identifier of the user that requested the page in which will be rendered the
   * widget.
   *
   * @param userId the user identifier.
   */
  public void setUserId(String userId) {
    this.userId = userId;
  }

  /**
   * Sets the language that requested the page in which will be rendered the
   * widget.
   *
   * @param language the user identifier.
   */
  public void setLanguage(String language) {
    this.language = language;
  }

  /**
   * Gets the unique identifier of the resource in Silverpeas.
   *
   * @return the resource identifier.
   */
  public String getResourceId() {
    return resourceId;
  }

  /**
   * Gets the type of the resource.
   *
   * @return
   */
  public String getResourceType() {
    return resourceType;
  }

  /**
   * Gets the unique identifier of the user that requested the page in which will be rendered the
   * widget.
   *
   * @return the unique identifier of the user.
   */
  public String getUserId() {
    return userId;
  }

  /**
   * Gets the language that requested the page in which will be rendered the
   * widget.
   *
   * @return the language.
   */
  public String getLanguage() {
    return language;
  }

  /**
   * Gets the html code generated by the widget.
   *
   * @return the code.
   */
  public String print() {

    StringBuilder builder = new StringBuilder(100);

    String dateReminder = "";
    String messageReminder = "";
    EntityReference resource = null;

    if(PublicationDetail.getResourceType().equals(getResourceType())) {//The resource is a publication
      if(this.getResourceId() != null) {//Case of updating an existing publication
        resource = new PublicationNoteReference(this.getResourceId());
      }
    }

    if (resource != null) {//Case of updating an existing publication
      PersistentDateReminderService dateReminderService = DateReminderServiceProvider.getDateReminderService();
      PersistentResourceDateReminder persistentResourceDateReminder = dateReminderService.get(resource);
      if (persistentResourceDateReminder.isDefined()) {
        DateReminderDetail dateReminderDetail = persistentResourceDateReminder.getDateReminder();
        Date dateReminderDate = dateReminderDetail.getDateReminder();

        //initialize the date reminder and message to be displayed
        dateReminder = DateUtil.dateToString(dateReminderDate, getLanguage());
        messageReminder = dateReminderDetail.getMessage();
      }
    }

    builder.append("<fieldset id=\"reminder\" class=\"skinFieldset\">").append("\n");
    builder.append("<legend>").append(this.messages.getString("dateReminder.fieldSet")).append("</legend>").append("\n");

    builder.append("<div class=\"fields\">").append("\n");

    builder.append("<div class=\"field\" id=\"dateReminderArea\">").append("\n");
    builder.append("<label for=\"dateReminder\" class=\"txtlibform\">").append(
        this.messages.getString("dateReminder.date")).append("</label>").append("\n");
    builder.append("<div class=\"champs\">").append("\n");
    builder.append("<input id=\"dateReminder\" type=\"text\" class=\"dateToPick\" " +
        "name=\"DateReminder\" value=\"").append(dateReminder).append("\" size=\"12\" " +
        "maxlength=\"10\"/>").append("\n");
    builder.append("</div>").append("\n");
    builder.append("</div>").append("\n");

    builder.append("<div class=\"field\" id=\"messageReminderArea\">").append("\n");
    builder.append("<label for=\"messageReminder\" class=\"txtlibform\">").append(
        this.messages.getString("dateReminder.message")).append("</label>").append("\n");
    builder.append("<div class=\"champs\">").append("\n");
    builder.append("<textarea rows=\"4\" cols=\"65\" name=\"MessageReminder\" " +
        "id=\"messageReminder\">").append(messageReminder).append("</textarea>").append("\n");
    builder.append("</div>").append("\n");
    builder.append("</div>").append("\n");

    builder.append("</div>").append("\n");

    builder.append("</fieldset>").append("\n");

    return builder.toString();
  }
}
